#include <MacTypes.h>
#include <HID.h>

/*!
@header IOWarriorLib
The IOWarrior library provides convinience funtions to access the IOWarrior under Mac OS 9.
It hides all the complexity of finding and accessing USB devices from the programmer.

You will need to include the following Libraries in your project to use this software:
HIDLib, USBManagerLib and USBServicesLib.
 
*/

// IOWarrior interface linked list note
struct IOWarriorListNodeStruct
{
	HIDDeviceDispatchTablePtr	HIDDispatchTable;
	HIDDeviceConnectionRef		connectionRef;
	unsigned char				isOpen;
	struct IOWarriorListNodeStruct* nextNode;
	unsigned int				interfaceType;
};

typedef struct IOWarriorListNodeStruct IOWarriorListNode;


/*!
@defined kIOWarrior40Interface0
@discussion Interface type indentifier for an interface 0 of an IOWarrior 40.
*/
#define kIOWarrior40Interface0 0

/*!
@defined kIOWarrior40Interface1
 @discussion Interface type indentifier for an interface 1 of an IOWarrior 40.
 */
#define kIOWarrior40Interface1 1

/*!
@defined kIOWarrior24Interface0
 @discussion Interface type indentifier for an interface 0 of an IOWarrior 24.
 */
#define kIOWarrior24Interface0 2

/*!
@defined kIOWarrior24Interface1
 @discussion Interface type indentifier for an interface 0 of an IOWarrior 24.
 */
#define kIOWarrior24Interface1 3

/*!
@function IOWarriorInit
@abstract Initializes the IOWarrior library. Call this before calling any other functions.
@result Returns 0 if successfull, otherwise return value is different from 0.
 */
int IOWarriorInit ();

/*!
@function IOWarriorClose
@abstract Performs some cleanup. You should call this function when you application is about to quit.
 */
void IOWarriorClose ();

/*!
@function IOWarriorIsPresent
@abstract Use this function to too see of an IOWarrior device is present.
@result Returns 1 if at least one IOWarrior is connected to this system. Returns 0 if no IOWarrior device could be discovered.
 */
int IOWarriorIsPresent ();

/*!
@function IOWarriorWriteInterface0
@abstract Writes a 4 byte buffer (32 bit) to interface 0 of the first IOWarrior40 encountered.
@discussion The first byte of the passed buffer becomes mapped to the pins of port 0, the second byte to the pins of port 1 and so on.
@param inData A pointer to the data being written. Should be at least 4 bytes long.
@result Returns 0 if writing operation was successfull. Returns a value different from 0 otherwise.
 */
int IOWarriorWriteInterface0 (void *inData);

/*!
@function IOWarriorReadInterface0
@abstract Reads 4 byte (32 bit) from interface 0 of the first IOWarrior40 encountered.
 @param outData A pointer to a buffer where the data should be stored. Buffer has to be at least 4 bytes long
 @result Returns 0 if reading operation was successfull. Returns a value different from 0 otherwise.
*/
int IOWarriorReadInterface0 (void *outData);

/*!
@function IOWarriorWriteInterface1
 @abstract Writes a 7 byte output report to interface 1 of the first IOWarrior40 encountered.
 @discussion Use the function to control IOWarrior special modes (like IIC and LCD mode) of the IOWarrior. See the IOWarrior datasheet for details.
 @param inReportID The report ID of the request. See the IO Warrior Datasheet for the appropriate values.
 @param inData A pointer to the data being written. Should be at least 7 bytes long.
 @result Returns 0 if writing operation was successfull. Returns a value different from 0 otherwise.
 */
int IOWarriorWriteInterface1 (int inReportID, void *inData);

/*!
@function IOWarriorInstallReportHandlerInterface1
	@abstract Install report handler for interface1 of first IOWarrior40.
 @discussion Install a report handler function that is invoked when report data is received from interface 1 of the first IOWarrior40 encountered. 
 @param inHandlerToInstall See Hid.h for a description of this callback function type.
 @param inHandlerRefCon A reference value thats passed to your report handler function when invoked.
*/
OSErr IOWarriorInstallReportHandlerInterface1 (HIDReportHandlerProcPtr inHandlerToInstall, UInt32 inHandlerRefCon);


/*!
@function IOWarriorRemoveReportHandlerInterface1
	@abstract Removes report handler.
	@discussion Removes any previously installed report handler installed by IOWarriorInstallReportHandlerInterface1.
*/

OSErr IOWarriorRemoveReportHandlerInterface1 ();

/*!
@function IOWarriorFirstInterfaceNodeOfType
	@abstract Returns a reference to the first IOWarriorListNode struct of a certain interface type.
	@discussion Use this function to obtain a reference to the IOWarriorListNode struct of the first interface of type
	inInterfaceType. 
	Use the result as a parameter for the IOWarriorRead and IOWarriorWrite functions.
	@param inInterfaceType The requested interface type. NULL if no such interface is available.
*/
IOWarriorListNode* IOWarriorFirstInterfaceNodeOfType (int inInterfaceType);

/*!
@function IOWarriorWrite
	@abstract Sends an HID report to an interface.
	@discussion Use this function to send an HID report with a certain id and a certain size to the interface at
	inNode.
	@param inNode The interface the report data should be written to. Can be obtained by calling IOWarriorFirstInterfaceNodeOfType.
	@param inReportID The id of report to be written.
	@param inSize The size of the report.
	@param inData Pointer to the data to be written.
	@result Returns noErr (0) if operation was successful. Otherwise returns an error code different from noErr.
*/
int IOWarriorWrite (IOWarriorListNode* inNode, int inReportID, int inSize, void* inData);

/*!
@function IOWarriorRead
	@abstract Reads an HID report from an interface.
	@discussion Use this function to read an HID report with a certain id and a certain size from the interface at
	inNode.
	@param inNode The interface the report data should be written from. Can be obtained by calling IOWarriorFirstInterfaceNodeOfType.
	@param inReportID The id of report to be read.
	@param inSize The size of the report.
	@param inData Pointer to the space where the report data can be stored.
	@result Returns noErr (0) if operation was successful. Otherwise returns an error code different from noErr.
*/

int IOWarriorRead (IOWarriorListNode* inNode, int inReportID, int inSize, void* outData);

/*!
@function IOWarriorSetAutoCloseInterfaces
	@abstract Toggles interface auto close mode.
	@discussion Use this function to change the way the IOWarriorRead and IOWarrriorWrite functions treat open 
	interfaces after the requested operation finished. Per default the interface is closed. Closing the interface
	breaks previously installed report handlers. If you are using report handler, you should invoked this function
	with inFlag set to false. 
	@param inFlag true: close interface after read/write operations, false: leave interface opened
*/
void IOWarriorSetAutoCloseInterfaces (int inFlag);

/*!
@function IOWarriorCloseAllInterfaces
	@abstract Closes all open IOWarriorInterfaces.
	@discussion If you previously used IOWarriorSetAutoCloseInterfaces to leave interfaces open, use this function to
	close interface that may be still open.
*/
void IOWarriorCloseAllInterfaces ();

/*!
@function IOWarriorInstallReportHandlerForInterfaceAtNode
	@abstract Install report handler for an interface.
	@discussion Installs a custom report handler for the interface specified by inNode. Your handler will be invoked when the interface sends a report
	in interrupt mode.
	@param inNode The interface node pointing to the interface.
	@param inHandlerToInstall See Hid.h for a description of this callback function type.
 	@param inHandlerRefCon A reference value thats passed to your report handler function when invoked.
 	@result An error code.
 */
OSErr IOWarriorInstallReportHandlerForInterfaceAtNode (IOWarriorListNode *inNode, HIDReportHandlerProcPtr inHandlerToInstall, UInt32 inHandlerRefCon);

/*!
@function IOWarriorRemoveReportHandlerForInterfaceAtNode
	@abstract Removes previously installed report handler.
	@discussion Removes any previously installed report handler for the interface specified by inNode.
	@param inNode The interface
	@result An error code.
*/
 
OSErr IOWarriorRemoveReportHandlerForInterfaceAtNode (IOWarriorListNode *inNode);
